/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.component.telegram.model.payments;

import java.io.Serial;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.JsonProperty;
import org.apache.camel.component.telegram.TelegramMessage;

/**
 * Refunds a successful payment in Telegram Stars. Returns True on success.
 *
 * @see <a href=
 *      "https://core.telegram.org/bots/api#refundstarpayment">https://core.telegram.org/bots/api#refundstarpayment</a>
 */
@JsonInclude(JsonInclude.Include.NON_NULL)
public class RefundStarPaymentMessage implements TelegramMessage {

    @Serial
    private static final long serialVersionUID = 5844786980495420170L;

    /**
     * Identifier of the user whose payment will be refunded.
     */
    @JsonProperty("user_id")
    private Long userId;

    /**
     * Telegram payment identifier.
     */
    @JsonProperty("telegram_payment_charge_id")
    private String telegramPaymentChargeId;

    public RefundStarPaymentMessage() {
    }

    public RefundStarPaymentMessage(Long userId, String telegramPaymentChargeId) {
        this.userId = userId;
        this.telegramPaymentChargeId = telegramPaymentChargeId;
    }

    public Long getUserId() {
        return userId;
    }

    public void setUserId(Long userId) {
        this.userId = userId;
    }

    public String getTelegramPaymentChargeId() {
        return telegramPaymentChargeId;
    }

    public void setTelegramPaymentChargeId(String telegramPaymentChargeId) {
        this.telegramPaymentChargeId = telegramPaymentChargeId;
    }

    @Override
    public String toString() {
        final StringBuilder sb = new StringBuilder("RefundStarPaymentMessage{");
        sb.append("userId=").append(userId);
        sb.append(", telegramPaymentChargeId='").append(telegramPaymentChargeId).append('\'');
        sb.append('}');
        return sb.toString();
    }
}
