/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.camel.component.jackson;

import com.fasterxml.jackson.core.StreamReadConstraints;
import com.fasterxml.jackson.databind.ObjectMapper;
import org.apache.camel.spi.Metadata;
import org.apache.camel.spi.annotations.Dataformat;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * Marshal POJOs to JSON and back using Jackson.
 */
@Dataformat("jackson")
@Metadata(excludeProperties = "library,permissions,dateFormatPattern")
public class JacksonDataFormat extends AbstractJacksonDataFormat {

    private static final Logger LOG = LoggerFactory.getLogger(JacksonDataFormat.class);

    /**
     * Use the default Jackson {@link ObjectMapper} and {@link Object}
     */
    public JacksonDataFormat() {
    }

    /**
     * Use the default Jackson {@link ObjectMapper} and with a custom unmarshal type
     *
     * @param unmarshalType the custom unmarshal type
     */
    public JacksonDataFormat(Class<?> unmarshalType) {
        super(unmarshalType);
    }

    /**
     * Use the default Jackson {@link ObjectMapper} and with a custom unmarshal type and JSON view
     *
     * @param unmarshalType the custom unmarshal type
     * @param jsonView      marker class to specify properties to be included during marshalling. See also
     */
    public JacksonDataFormat(Class<?> unmarshalType, Class<?> jsonView) {
        super(unmarshalType, jsonView);
    }

    /**
     * Use a custom Jackson mapper and an unmarshal type
     *
     * @param mapper        the custom mapper
     * @param unmarshalType the custom unmarshal type
     */
    public JacksonDataFormat(ObjectMapper mapper, Class<?> unmarshalType) {
        super(mapper, unmarshalType);
    }

    /**
     * Use a custom Jackson mapper, unmarshal type and JSON view
     *
     * @param mapper        the custom mapper
     * @param unmarshalType the custom unmarshal type
     * @param jsonView      marker class to specify properties to be included during marshalling. See also
     */
    public JacksonDataFormat(ObjectMapper mapper, Class<?> unmarshalType, Class<?> jsonView) {
        super(mapper, unmarshalType, jsonView);
    }

    @Override
    public String getDataFormatName() {
        return "jackson";
    }

    @Override
    protected ObjectMapper createNewObjectMapper() {
        ObjectMapper om = new ObjectMapper();
        int len = getMaxStringLength();
        if (len > 0) {
            LOG.debug("Creating ObjectMapper with maxStringLength: {}", len);
            om.getFactory().setStreamReadConstraints(StreamReadConstraints.builder().maxStringLength(len).build());
        }
        return om;
    }

    @Override
    protected Class<? extends ObjectMapper> getObjectMapperClass() {
        return ObjectMapper.class;
    }

    @Override
    protected String getDefaultContentType() {
        return "application/json";
    }

}
